package pl.islandworld.commands;

import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.lang.StringUtils;
import org.bukkit.Bukkit;
import org.bukkit.ChatColor;
import org.bukkit.Chunk;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.OfflinePlayer;
import org.bukkit.World;
import org.bukkit.block.Biome;
import org.bukkit.block.Block;
import org.bukkit.command.Command;
import org.bukkit.command.CommandExecutor;
import org.bukkit.command.CommandSender;
import org.bukkit.entity.Entity;
import org.bukkit.entity.Monster;
import org.bukkit.entity.Player;

import pl.islandworld.Config;
import pl.islandworld.Eco;
import pl.islandworld.IslandWorld;
import pl.islandworld.entity.MyLocation;
import pl.islandworld.entity.SimpleIslandV6;

import com.sk89q.worldedit.schematic.SchematicFormat;

/**
 * 
 * @author Gnacik
 * 
 */
public class Island implements CommandExecutor
{
	private final IslandWorld plugin;

	private HashMap<String, Long> lastCommand = new HashMap<String, Long>();
	private HashMap<String, Long> lastCalc = new HashMap<String, Long>();

	public Island(IslandWorld plugin)
	{
		this.plugin = plugin;
	}

	@Override
	public boolean onCommand(CommandSender sender, Command command, String label, String[] split)
	{
		if (!(sender instanceof Player))
			return false;
		if (plugin == null)
			return false;
		final Player player = (Player) sender;
		if (IslandWorld.REGEN_IN_PROGRESS)
			return plugin.showError(player, plugin.getLoc("error-regen"));
		if (!IslandWorld.LOADED)
			return plugin.showError(player, plugin.getLoc("error-start"));
		final World world = plugin.getServer().getWorld(Config.WORLD_ISLE);
		if (world == null)
			return plugin.showError(player, plugin.getLoc("error-world"));
		if (plugin.isOnDeleteList(player) && split.length > 0 && !split[0].equalsIgnoreCase("delete"))
			return plugin.showError(player, plugin.getLoc("error-delete-comm"));

		if (split.length == 0)
		{
			showUsage(player);
		}
		else if (split.length == 1)
		{
			if (split[0].equalsIgnoreCase("create"))
			{
				return commandCreate(world, player, sender, null);
			}
			else if (split[0].equalsIgnoreCase("delete"))
			{
				return commandDelete(world, player);
			}
			else if (split[0].equalsIgnoreCase("info"))
			{
				return commandInfo(world, player);
			}
			else if (split[0].equalsIgnoreCase("home"))
			{
				return commandHome(world, player, null);
			}
			else if (split[0].equalsIgnoreCase("sethome"))
			{
				return commandSetHome(world, player, null);
			}
			else if (split[0].equalsIgnoreCase("delhome"))
			{
				return commandDelHome(world, player, null);
			}
			else if (split[0].equalsIgnoreCase("expell"))
			{
				return commandExpell(world, player, null);
			}
			else if (split[0].equalsIgnoreCase("add"))
			{
				return commandAdd(world, player, null);
			}
			else if (split[0].equalsIgnoreCase("remove"))
			{
				return commandRemove(world, player, null);
			}
			else if (split[0].equalsIgnoreCase("tp"))
			{
				return commandTp(world, player, null);
			}
			else if (split[0].equalsIgnoreCase("leave"))
			{
				return commandLeave(world, player);
			}
			else if (split[0].equalsIgnoreCase("spawn"))
			{
				return commandSpawn(world, player);
			}
			else if (split[0].equalsIgnoreCase("fixhome"))
			{
				return commandFixHome(world, player);
			}
			else if (split[0].equalsIgnoreCase("lock"))
			{
				return commandLock(world, player);
			}
			else if (split[0].equalsIgnoreCase("unlock"))
			{
				return commandUnLock(world, player);
			}
			else if (split[0].equalsIgnoreCase("visit"))
			{
				return commandVisit(world, player, null);
			}
			else if (split[0].equalsIgnoreCase("accept"))
			{
				return commandAccept(world, player);
			}
			else if (split[0].equalsIgnoreCase("join"))
			{
				return commandJoin(world, player);
			}
			else if (split[0].equalsIgnoreCase("deny"))
			{
				return commandDeny(world, player);
			}
			else if (split[0].equalsIgnoreCase("calc"))
			{
				return commandCalc(world, player);
			}
			else if (split[0].equalsIgnoreCase("rank"))
			{
				return commandRank(world, player);
			}
			else if (split[0].equalsIgnoreCase("biomelist"))
			{
				return commandBiomeList(world, player);
			}
			else if (split[0].equalsIgnoreCase("biomeshow"))
			{
				return commandBiomeShow(world, player);
			}
			else if (split[0].equalsIgnoreCase("biomeset"))
			{
				return commandBiomeSet(world, player, null, false);
			}
			else if (split[0].equalsIgnoreCase("biomechunkset"))
			{
				return commandBiomeSet(world, player, null, true);
			}
			else
				showUsage(player);
		}
		else if (split.length == 2)
		{
			if (split[0].equalsIgnoreCase("create"))
			{
				if (!(player.hasPermission("islandworld.create." + split[1].toLowerCase())))
					return plugin.showError(player, plugin.getLoc("error-no-perms"));

				commandCreate(world, player, sender, split[1]);
			}
			else if (split[0].equalsIgnoreCase("add"))
			{
				return commandAdd(world, player, split[1]);
			}
			else if (split[0].equalsIgnoreCase("remove"))
			{
				return commandRemove(world, player, split[1]);
			}
			else if (split[0].equalsIgnoreCase("tp"))
			{
				return commandTp(world, player, split[1]);
			}
			else if (split[0].equalsIgnoreCase("visit"))
			{
				return commandVisit(world, player, split[1]);
			}
			else if (split[0].equalsIgnoreCase("expell"))
			{
				return commandExpell(world, player, split[1]);
			}
			else if (split[0].equalsIgnoreCase("home"))
			{
				return commandHome(world, player, split[1]);
			}
			else if (split[0].equalsIgnoreCase("sethome"))
			{
				return commandSetHome(world, player, split[1]);
			}
			else if (split[0].equalsIgnoreCase("delhome"))
			{
				return commandDelHome(world, player, split[1]);
			}
			else if (split[0].equalsIgnoreCase("biomeset"))
			{
				return commandBiomeSet(world, player, split[1], false);
			}
			else if (split[0].equalsIgnoreCase("biomechunkset"))
			{
				return commandBiomeSet(world, player, split[1], true);
			}
			else
				showUsage(player);
		}
		else
			showUsage(player);

		return true;
	}

	private boolean commandBiomeSet(World world, Player player, String biome, boolean chunkOnly)
	{
		if (!(player.hasPermission(chunkOnly ? "islandworld.island.biomechunkset" : "islandworld.island.biomeset")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));
		if (biome == null || biome.isEmpty())
			return plugin.showError(player, plugin.getLoc("error-invalid-biome").replaceAll("%biome%", ""));
		if (Config.BIOME_PERMISSION && !player.hasPermission("islandworld.island.biomeset." + biome.toLowerCase()))
			return plugin.showError(player, plugin.getLoc("error-no-perms-biome"));

		SimpleIslandV6 island = plugin.getPlayerIsland(player.getName());
		if (island != null)
		{
			if (plugin.isInsideIsland(player, island))
			{
				Biome newbiome = null;
				Biome[] bm = Biome.values();
				for (Biome b : bm)
				{
					if (b.toString().equalsIgnoreCase(biome))
						newbiome = b;
				}

				if (newbiome != null)
				{
					if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, chunkOnly ? "biomechunkset" : "biomeset"))
						return plugin.showError(
								player,
								plugin.getLoc("eco-failed").replaceAll("%cost%",
										String.valueOf(plugin.getConfig().getInt("command-cost." + (chunkOnly ? "biomechunkset" : "biomeset"), 0))));

					final Chunk chunk = world.getChunkAt(player.getLocation());

					if (chunkOnly)
					{
						if (chunk != null)
						{
							final int startx = chunk.getX() * 16;
							final int startz = chunk.getZ() * 16;

							for (int x = startx; x < (startx + 16); x++)
							{
								for (int z = startz; z < (startz + 16); z++)
								{
									plugin.getIslandWorld().setBiome(x, z, newbiome);
								}
							}
						}
					}
					else
					{
						int x = island.getX() * Config.ISLE_SIZE;
						int z = island.getZ() * Config.ISLE_SIZE;

						for (int x_operate = x; x_operate < (x + Config.ISLE_SIZE); x_operate++)
						{
							for (int z_operate = z; z_operate < (z + Config.ISLE_SIZE); z_operate++)
							{
								plugin.getIslandWorld().setBiome(x_operate, z_operate, newbiome);
							}
						}
					}
					if (chunkOnly)
						player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-biome-chunk-changed").replaceAll("%biome%", newbiome.toString()));
					else
						player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-biome-changed").replaceAll("%biome%", newbiome.toString()));

					if (chunk != null)
					{
						if (!world.isChunkLoaded(chunk))
							world.loadChunk(chunk);
						world.refreshChunk(chunk.getX(), chunk.getZ());
					}
				}
				else
					player.sendMessage(ChatColor.RED + plugin.getLoc("error-invalid-biome").replaceAll("%biome%", biome));
			}
			else
				player.sendMessage(ChatColor.RED + plugin.getLoc("error-outside-island"));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return false;
	}

	private boolean commandBiomeList(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.biomelist")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "biomelist"))
			return plugin.showError(player,
					plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.biomelist", 0))));

		Biome[] bm = Biome.values();

		player.sendMessage(ChatColor.AQUA + plugin.getLoc("info-biomes"));
		player.sendMessage(ChatColor.GREEN + "" + StringUtils.join(bm, ", "));

		return false;
	}

	private boolean commandBiomeShow(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.biomeshow")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		SimpleIslandV6 island = plugin.getPlayerIsland(player.getName());
		if (island != null)
		{
			if (plugin.isInsideIsland(player, island))
			{
				Biome bio = player.getWorld().getBiome(player.getLocation().getBlockX(), player.getLocation().getBlockZ());
				if (bio != null)
				{
					if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "biomeshow"))
						return plugin.showError(
								player,
								plugin.getLoc("eco-failed").replaceAll("%cost%",
										String.valueOf(plugin.getConfig().getInt("command-cost.biomeshow", 0))));

					player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-biome").replaceAll("%biome%", bio.toString()));
					return true;
				}
			}
			else
				player.sendMessage(ChatColor.RED + plugin.getLoc("error-outside-island"));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return false;
	}

	private boolean commandCreate(World world, Player player, CommandSender sender, String schematicName)
	{
		if (player == null)
			return false;
		if (!(player.hasPermission("islandworld.island.create")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		if (plugin.haveIsland(player))
			return plugin.showError(player, plugin.getLoc("error-have-island"));
		if (Config.STRICT_PARTY && plugin.isHelping(player))
			return plugin.showError(player, plugin.getLoc("error-helping"));

		if (plugin.getFreeList() != null && !plugin.getFreeList().isEmpty())
		{
			SimpleIslandV6 newIsland = plugin.getFreeList().iterator().next();

			if (newIsland != null)
			{
				if (schematicName == null || schematicName.isEmpty())
					schematicName = (player.hasPermission("islandworld.vip.schematic") && plugin.getSchematicFromFile(IslandWorld.DESIGN_SPECIAL,
							true) != null) ? IslandWorld.DESIGN_SPECIAL : IslandWorld.DESIGN_NORMAL;

				final SchematicFormat schema = plugin.getSchematicFromFile(schematicName, false);
				if (schema != null)
				{
					if (!(player.hasPermission("islandworld.create." + schematicName)))
						return plugin.showError(player, plugin.getLoc("error-schematic").replaceAll("%name%", schematicName));

					if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "create"))
						return plugin
								.showError(
										player,
										plugin.getLoc("eco-failed").replaceAll("%cost%",
												String.valueOf(plugin.getConfig().getInt("command-cost.create", 0))));

					// Clean inventory only if player is in Island World
					if (Config.CLEANINV && player.getWorld() == world)
						player.getInventory().clear();

					// Create Island
					plugin.onCreate(newIsland, player, schematicName);
					// Send info
					player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-created"));
					// Save Debug
					plugin.writeLog(newIsland, "Created by player " + sender.getName());
				}
				else
					player.sendMessage(ChatColor.RED + plugin.getLoc("error-schematic-not-exists").replaceAll("%name%", schematicName));
			}
			else
				player.sendMessage(ChatColor.RED + plugin.getLoc("error-get-free"));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-free"));

		return true;
	}

	private boolean commandDelete(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.delete")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		if (player.getWorld() != world)
			return plugin.showError(player, plugin.getLoc("error-current-world"));

		if (plugin.haveIsland(player))
		{
			if (playerHaveCommandLimit(player))
				return plugin.showError(player, plugin.getLoc("error-delay").replaceAll("%time%", String.valueOf(Config.TIME_LIMIT)));

			if (plugin.isOnDeleteList(player))
			{
				if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "delete"))
					return plugin.showError(player,
							plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.delete", 0))));

				// Clean inventory
				if (Config.CLEANINV)
					player.getInventory().clear();

				// Clean challenges
				if (plugin.getConfig().getBoolean("delete-challenges", false))
					plugin.deleteChallenges(player);

				// Get correct isle
				SimpleIslandV6 isle = plugin.getPlayerIsland(player);
				// All done
				player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-deleted"));
				// Teleport all player outside
				Player[] players = Bukkit.getOnlinePlayers();
				for (Player pl : players)
				{
					if (pl != null && plugin.isInsideIsland(pl, isle))
					{
						pl.teleport(plugin.getSpawnWorld().getSpawnLocation());
					}
				}
				// Set home
				if (Config.SET_HOME)
					player.performCommand("sethome");
				// Remove from list
				plugin.removeFromDeleteList(player);
				// Set limit
				lastCommand.put(player.getName().toLowerCase(), System.currentTimeMillis());
				// Delete island
				plugin.onDelete(isle, player.getName().toLowerCase());
				// Save Debug
				plugin.writeLog(isle, "Deleted by user " + player.getName());
			}
			else
				plugin.addToDeleteList(player);
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return true;
	}

	private boolean commandInfo(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.info")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		if (plugin.haveIsland(player))
		{
			if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "info"))
				return plugin.showError(player,
						plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.info", 0))));

			// Get correct isle
			SimpleIslandV6 isle = plugin.getPlayerIsland(player);
			// Show info
			plugin.showInfo(player, player.getName(), isle);
		}
		else
		{
			// Get correct isle
			final SimpleIslandV6 isle = plugin.getHelpingIsland(player);
			// Teleport
			if (isle != null)
				plugin.showInfo(player, player.getName(), isle);
			else
				player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));
		}

		return true;
	}

	private boolean commandHome(World world, Player player, String name)
	{
		if (!(player.hasPermission("islandworld.island.home")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));
		if (plugin.getConfig().getBoolean("falling-block-tp", false) && plugin.isFalling(player))
			return plugin.showError(player, plugin.getLoc("error-tp-while-falling"));

		SimpleIslandV6 isle = plugin.getPlayerIsland(player);
		if (isle == null)
			isle = plugin.getHelpingIsland(player);

		// Teleport
		if (isle != null)
		{
			if (name == null || name.isEmpty())
			{
				MyLocation dest_loc = isle.getLocation();
				if (dest_loc != null)
				{
					if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "home"))
						return plugin.showError(player, plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.home", 0))));
					if (!plugin.isSafeToTeleport(dest_loc))
						return plugin.showError(player, plugin.getLoc("error-tp-not-safe"));
					
					removeMobs(dest_loc);

					final int tp_delay = plugin.getConfig().getInt("teleport-delay", 0);
					if (tp_delay > 0 && !player.hasPermission("islandworld.bypass.tpdelay"))
					{
						player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-teleport-after").replaceAll("%time%", String.valueOf(tp_delay)));
						final int taskId = Bukkit.getScheduler().scheduleSyncDelayedTask(plugin, new FinishTeleport(plugin.getIslandWorld(), player, dest_loc), 20 * tp_delay);

						plugin.addToTeleportList(player, taskId);
					}
					else
					{
						plugin.teleportPlayer(plugin.getIslandWorld(), player, dest_loc);
						plugin.showMessage(player, plugin.getLoc("info-teleported"));
					}
				}
				else
					plugin.showError(player, plugin.getLoc("error-is-no-home"));
			}
			else if (name.equalsIgnoreCase("list"))
			{
				player.sendMessage(ChatColor.GREEN + "Your homes: " + ChatColor.YELLOW + StringUtils.join(isle.getHomes().keySet().toArray(), ", "));
			}
			else
			{
				final MyLocation dest_loc = isle.getHome(name);
				if (dest_loc != null)
				{
					final int tp_delay = plugin.getConfig().getInt("teleport-delay", 0);
					if (tp_delay > 0 && !player.hasPermission("islandworld.bypass.tpdelay"))
					{
						player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-teleport-after").replaceAll("%time%", String.valueOf(tp_delay)));
						final int taskId = Bukkit.getScheduler().scheduleSyncDelayedTask(plugin, new FinishTeleport(plugin.getIslandWorld(), player, dest_loc), 20 * tp_delay);
						
						if (plugin.getConfig().getBoolean("movement-break-tp", false))
							plugin.addToTeleportList(player, taskId);
					}
					else
					{
						plugin.teleportPlayer(plugin.getIslandWorld(), player, dest_loc);
						plugin.showMessage(player, plugin.getLoc("info-teleported"));
					}
				}
				else
					plugin.showError(player, plugin.getLoc("error-no-home").replaceAll("%name%", name));
			}
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return true;
	}

	private boolean commandDelHome(World world, Player player, String name)
	{
		if (!(player.hasPermission("islandworld.island.delhome")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		if (name == null || name.isEmpty())
		{
			player.sendMessage(ChatColor.AQUA + "Usage:\n");
			player.sendMessage(ChatColor.RED + "/island delhome <name>" + ChatColor.AQUA + " - delete home location\n");
			return false;
		}
		else
		{
			if (plugin.haveIsland(player))
			{
				if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "delhome"))
					return plugin.showError(player,
							plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.delhome", 0))));

				// Get correct isle
				SimpleIslandV6 isle = plugin.getPlayerIsland(player);
				// Remove it
				isle.removeHome(name);
				// Show info
				player.sendMessage(plugin.getLoc("info-home-deleted").replaceAll("%name%", name));
			}
			else
				player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));
		}

		return true;
	}

	private boolean commandSetHome(World world, Player player, String name)
	{
		if (!(player.hasPermission("islandworld.island.sethome")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		if (plugin.haveIsland(player))
		{
			if (plugin.isInsideOwnIsland(player))
			{
				// Get correct isle
				SimpleIslandV6 isle = plugin.getPlayerIsland(player);
				// Get coords
				final Location loc = player.getLocation();
				if (name == null || name.isEmpty())
				{
					if (plugin.isSafeToTeleport(new MyLocation(player.getLocation())))
					{					
						if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "sethome"))
							return plugin.showError(player,plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.sethome", 0))));
	
						// Set it
						isle.setLocation(loc);
						// Show info
						player.sendMessage(plugin.getLoc("info-loc-changed"));
						// Show debug
						plugin.debug("Home Location changed to : [" + loc.getBlockX() + "][" + loc.getBlockY() + "][" + loc.getBlockZ() + "]");
					}
					else
						return plugin.showError(player, plugin.getLoc("error-sethome-loc"));						
				}
				else
				{
					// Get limit
					final int norLimit = plugin.getConfig().getInt("home-limit", 0);
					final int vipLimit = plugin.getConfig().getInt("home-limit-vip", 0);
					// Check
					final int hLimit = player.hasPermission("islandworld.vip.homes") ? vipLimit : norLimit;

					if (hLimit > 0)
					{
						if (isle.getHomes().size() >= hLimit && isle.getHome(name) == null)
						{
							return plugin.showError(player, plugin.getLoc("error-home-limit").replaceAll("%count%", String.valueOf(hLimit)));
						}
						else
						{
							if (plugin.isSafeToTeleport(new MyLocation(player.getLocation())))
							{
								if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "sethome"))
									return plugin.showError(player, plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.sethome", 0))));

								// Set it
								isle.addHome(name, new MyLocation(loc));
								// Show info
								player.sendMessage(plugin.getLoc("info-home-set"));
								// Show debug
								plugin.debug("Home (" + name + ") location changed to : [" + loc.getBlockX() + "][" + loc.getBlockY() + "]["+ loc.getBlockZ() + "]");
							}
							else
								return plugin.showError(player, plugin.getLoc("error-sethome-loc"));
						}
					}
				}
			}
			else
				plugin.showError(player, plugin.getLoc("error-outside"));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return true;
	}

	private boolean commandExpell(World world, Player player, String plName)
	{
		if (!(player.hasPermission("islandworld.island.expell")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		final SimpleIslandV6 island = plugin.getPlayerIsland(player);
		if (island != null)
		{
			if (plName == null || plName.isEmpty())
			{
				if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "expell"))
					return plugin.showError(player,
							plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.expell", 0))));

				int count = plugin.expellPlayers(player, island);

				if (count > 0)
					player.sendMessage(plugin.getLoc("info-expell").replaceAll("%count%", String.valueOf(count)));
				else
					player.sendMessage(plugin.getLoc("error-expell"));
			}
			else
			{
				if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "expell"))
					return plugin.showError(player,
							plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.expell", 0))));

				Player pl = Bukkit.getPlayer(plName);
				if (pl != null && !pl.isOp())
				{
					if (plugin.isInsideIsland(pl, island))
					{
						pl.sendMessage(ChatColor.AQUA + plugin.getLoc("info-expelled").replaceAll("%name%", player.getName()));
						pl.teleport(plugin.getSpawnWorld().getSpawnLocation());

						player.sendMessage(plugin.getLoc("info-expell").replaceAll("%count%", "1"));
					}
					else
						player.sendMessage(plugin.getLoc("error-expell-outside"));
				}
				else
					player.sendMessage(plugin.getLoc("error-expell"));
			}
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return true;
	}

	private boolean commandAdd(World world, Player player, String plName)
	{
		if (!(player.hasPermission("islandworld.island.add")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		if (plName == null || plName.isEmpty())
		{
			player.sendMessage(ChatColor.AQUA + "Usage:\n");
			player.sendMessage(ChatColor.RED + "/island add <playerName>" + ChatColor.AQUA + " - add player to our isle.\n");
			return false;
		}

		if (plugin.haveIsland(player))
		{
			final SimpleIslandV6 island = plugin.getPlayerIsland(player);
			final Player friend = Bukkit.getPlayer(plName);
			if (island != null && friend != null && friend.isOnline() && player.canSee(friend))
			{
				if (Config.STRICT_PARTY && plugin.haveIsland(friend))
					return plugin.showError(player, plugin.getLoc("error-add-have-island"));
				if (Config.STRICT_PARTY && plugin.isHelping(friend))
					return plugin.showError(player, plugin.getLoc("error-add-have-party"));

				if (plugin.getPartyList().containsKey(player.getName().toLowerCase()))
					return plugin.showError(player, plugin.getLoc("error-party-visitor-wait"));

				if (plugin.getPartyList().containsValue(friend.getName().toLowerCase()))
					return plugin.showError(player, plugin.getLoc("error-party-owner-wait").replaceAll("%name%", friend.getName()));
				
				// Get limit
				final int norLimit = plugin.getConfig().getInt("party-limit", 0);
				final int vipLimit = plugin.getConfig().getInt("party-limit-vip", 0);
				// Check
				final int pLimit = player.hasPermission("islandworld.vip.party") ? vipLimit : norLimit;
				// Members
				final List<String> members = island.getMembers();
				if (members != null)
				{
					if (pLimit > 0 && members.size() >= pLimit)
						return plugin.showError(player, plugin.getLoc("error-add-limit"));

					if (!members.contains(plName.toLowerCase()))
					{
						if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "add"))
							return plugin.showError(player,plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.add", 0))));

						plugin.makePartyRequest(player, friend);
						
						player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-party-request").replaceAll("%name%", friend.getName()));
						friend.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-party-need-answer").replaceAll("%name%", player.getName()));
						
						//island.addMember(plName.toLowerCase());
						//plugin.setIsHelping(plName, island);
					}
					else
						player.sendMessage(ChatColor.RED + plugin.getLoc("error-add-already").replaceAll("%name%", plName));
				}
				else
					plugin.showError(player, plugin.getLoc("error-wg-region"));
			}
			else
				plugin.showError(player, plugin.getLoc("error-not-online").replaceAll("%name%", plName));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return true;
	}

	private boolean commandRemove(World world, Player player, String plName)
	{
		if (!(player.hasPermission("islandworld.island.remove")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		if (plName == null || plName.isEmpty())
		{
			player.sendMessage(ChatColor.AQUA + "Usage:\n");
			player.sendMessage(ChatColor.RED + "/island remove <playerName>" + ChatColor.AQUA + " - remove player from your isle.\n");
			return false;
		}

		final SimpleIslandV6 island = plugin.getPlayerIsland(player);
		if (island != null)
		{
			final String friendName = plName.toLowerCase();
			// Check
			final List<String> members = island.getMembers();
			if (members != null && members.contains(friendName))
			{
				if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "remove"))
					return plugin.showError(player, plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.remove", 0))));

				island.removeMember(friendName);
				player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-remove-ok").replaceAll("%name%", friendName));
				plugin.removeHelping(friendName);
				final Player friend = Bukkit.getPlayer(friendName);
				if (friend != null && plugin.isInsideIsland(friend, island))
				{
					// Teleport player
					friend.teleport(plugin.getSpawnWorld().getSpawnLocation());
					// Send message
					friend.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-removed-ok").replaceAll("%name%", player.getName()));
				}
			}
			else
				player.sendMessage(ChatColor.RED + plugin.getLoc("error-remove-not-added").replaceAll("%name%", friendName));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return false;
	}

	private boolean commandTp(World world, Player player, String plName)
	{
		if (!(player.hasPermission("islandworld.island.tp")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));
		if (plugin.getConfig().getBoolean("falling-block-tp", false) && plugin.isFalling(player))
			return plugin.showError(player, plugin.getLoc("error-tp-while-falling"));

		if (plName == null || plName.isEmpty())
		{
			player.sendMessage(ChatColor.AQUA + "Usage:\n");
			player.sendMessage(ChatColor.RED + "/island tp <playerName>" + ChatColor.AQUA + " - teleport to player isle.\n");
			return false;
		}

		final OfflinePlayer target = Bukkit.getOfflinePlayer(plName);
		if (target != null)
		{
			if (Config.STRICT_PARTY && plugin.haveIsland(player))
				return plugin.showError(player, plugin.getLoc("error-tp-have-island"));

			SimpleIslandV6 isle = plugin.getPlayerIsland(target.getName());
			if (isle != null)
			{
				final List<String> members = isle.getMembers();
				if (members != null && members.contains(player.getName().toLowerCase()))
				{
					if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "tp"))
						return plugin.showError(player,
								plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.tp", 0))));

					final MyLocation dest_loc = isle.getLocation();
					final int tp_delay = plugin.getConfig().getInt("teleport-delay", 0);
					if (tp_delay > 0 && !player.hasPermission("islandworld.bypass.tpdelay"))
					{
						player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-teleport-after").replaceAll("%time%", String.valueOf(tp_delay)));
						final int taskId = Bukkit.getScheduler().scheduleSyncDelayedTask(plugin, new FinishTeleport(plugin.getIslandWorld(), player, dest_loc), 20 * tp_delay);

						plugin.addToTeleportList(player, taskId);
					}
					else
					{
						plugin.teleportPlayer(plugin.getIslandWorld(), player, dest_loc);
						player.sendMessage(plugin.getLoc("info-tp-to-island").replaceAll("%name%", plName));
					}
				}
				else
					player.sendMessage(ChatColor.RED + plugin.getLoc("error-tp-no-party").replaceAll("%name%", target.getName()));
			}
			else
				player.sendMessage(ChatColor.RED + plugin.getLoc("error-tp-no-island").replaceAll("%name%", target.getName()));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-player"));

		return true;
	}

	private boolean commandVisit(World world, Player player, String plName)
	{
		if (!(player.hasPermission("islandworld.island.visit")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));
		if (!plugin.getConfig().getBoolean("allow-visit", false))
			return plugin.showError(player, plugin.getLoc("error-no-visit"));
		if (plugin.getConfig().getBoolean("falling-block-tp", false) && plugin.isFalling(player))
			return plugin.showError(player, plugin.getLoc("error-tp-while-falling"));

		if (plName == null || plName.isEmpty())
		{
			player.sendMessage(ChatColor.AQUA + "Usage:\n");
			player.sendMessage(ChatColor.RED + "/island visit <playerName>" + ChatColor.AQUA + " - make visit request.\n");
			return false;
		}
		final Player target = Bukkit.getPlayer(plName);
		if (target != null && player.canSee(target))
		{
			if (target.getName().equalsIgnoreCase(player.getName()))
				return plugin.showError(player, plugin.getLoc("error-visit-self"));

			SimpleIslandV6 isle = plugin.getPlayerIsland(target.getName());
			if (isle != null)
			{
				if (plugin.getVisitList().containsKey(target.getName().toLowerCase()))
					return plugin.showError(player, plugin.getLoc("error-visit-owner-wait").replaceAll("%name%", target.getName()));

				if (plugin.getVisitList().containsValue(player.getName().toLowerCase()))
					return plugin.showError(player, plugin.getLoc("error-visit-visitor-wait"));

				if (targetIsSafePoint(world, isle))
				{
					if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "visit"))
						return plugin.showError(player, plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.visit", 0))));

					plugin.makeVisitRequest(target, player);
					
					player.sendMessage(ChatColor.AQUA + plugin.getLoc("info-visit-request").replaceAll("%name%", target.getName()));
					target.sendMessage(ChatColor.AQUA + plugin.getLoc("info-visit-need-answer").replaceAll("%name%", player.getName()));
				}
				else
				{
					player.sendMessage(ChatColor.RED + plugin.getLoc("error-visit-spawn").replaceAll("%name%", target.getName()));
					target.sendMessage(ChatColor.RED + plugin.getLoc("error-visit-your-spawn").replaceAll("%name%", player.getName()));
				}
			}
			else
				player.sendMessage(ChatColor.RED + plugin.getLoc("error-visit-no-island").replaceAll("%name%", target.getName()));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-visit-no-player"));

		return true;
	}

	private boolean commandAccept(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.accept")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		final HashMap<String, String> tmpMap = new HashMap<String, String>();
		tmpMap.putAll(plugin.getVisitList());

		SimpleIslandV6 isle = plugin.getPlayerIsland(player.getName());
		if (isle != null)
		{
			for (Entry<String, String> x : tmpMap.entrySet())
			{
				final String ow = x.getKey();
				final String vi = x.getValue();
				if (ow != null && ow.equalsIgnoreCase(player.getName().toLowerCase()))
				{
					plugin.getVisitList().remove(ow);

					final Player visitor = Bukkit.getPlayer(vi);
					if (visitor != null)
					{
						if (targetIsSafePoint(world, isle))
						{
							plugin.teleportPlayer(plugin.getIslandWorld(), visitor, isle.getLocation());
							visitor.sendMessage(plugin.getLoc("info-visit-visitor").replaceAll("%name%", player.getName()));
							player.sendMessage(plugin.getLoc("info-visit-visitor-tp").replaceAll("%name%", visitor.getName()));
						}
						else
						{
							visitor.sendMessage(ChatColor.RED + plugin.getLoc("error-visit-spawn").replaceAll("%name%", player.getName()));
							player.sendMessage(ChatColor.RED + plugin.getLoc("error-visit-your-spawn").replaceAll("%name%", visitor.getName()));
						}
					}
				}
			}
		}
		else
			plugin.showError(player, plugin.getLoc("error-no-island"));

		return true;
	}

	private boolean commandJoin(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.join")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		final HashMap<String, String> tmpMap = new HashMap<String, String>();
		tmpMap.putAll(plugin.getPartyList());

		for (Entry<String, String> x : tmpMap.entrySet())
		{
			final String ow = x.getKey();
			final String vi = x.getValue();
			
			if (ow != null && vi != null && vi.equalsIgnoreCase(player.getName().toLowerCase()))
			{
				plugin.getPartyList().remove(ow);

				final SimpleIslandV6 island = plugin.getPlayerIsland(ow);
				if (island != null)
				{
					final Player visitor = Bukkit.getPlayer(vi);
					if (visitor != null)
					{
						island.addMember(visitor.getName().toLowerCase());
						plugin.setIsHelping(visitor.getName(), island);
						
						visitor.sendMessage(plugin.getLoc("info-party-joined").replaceAll("%name%", ow));
						
						Player owner = Bukkit.getPlayer(ow);
						if (owner != null)
							owner.sendMessage(plugin.getLoc("info-party-joined-owner").replaceAll("%name%", visitor.getName()));
					}
				}
			}
		}

		return true;
	}
	
	private boolean commandDeny(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.deny")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		String visitor = plugin.getVisitList().get(player.getName().toLowerCase());

		if (visitor != null)
		{
			plugin.getVisitList().remove(player.getName().toLowerCase());

			final Player visi = Bukkit.getPlayer(visitor);
			if (visi != null)
				plugin.showError(visi, plugin.getLoc("error-visit-request-deny").replaceAll("%name%", player.getName()));

			plugin.showError(player, plugin.getLoc("error-visit-request-you-deny").replaceAll("%name%", visitor));
		}

		return true;
	}

	private boolean commandLeave(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.leave")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		SimpleIslandV6 island = plugin.getHelpingIsland(player);
		if (island != null)
		{
			if (plugin.getConfig().getBoolean("clean-inventory-leave", false))
				player.getInventory().clear();

			if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "leave"))
				return plugin.showError(player,
						plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.leave", 0))));

			// Remove from helping list
			plugin.removeHelping(player.getName());
			// Remove from island
			island.removeMember(player.getName());
			// Show info
			player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-leave-ok").replaceAll("%name%", island.getOwner()));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-leave"));

		return true;
	}

	private boolean commandSpawn(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.spawn")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));
		if (plugin.getConfig().getBoolean("falling-block-tp", false) && plugin.isFalling(player))
			return plugin.showError(player, plugin.getLoc("error-tp-while-falling"));

		// Clean inventory
		if (plugin.getConfig().getBoolean("clean-inventory-exit", false))
			player.getInventory().clear();

		if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "spawn"))
			return plugin.showError(player,
					plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.spawn", 0))));

		final MyLocation dest_loc = new MyLocation(plugin.getSpawnWorld().getSpawnLocation());
		final int tp_delay = plugin.getConfig().getInt("teleport-delay", 0);
		if (tp_delay > 0 && !player.hasPermission("islandworld.bypass.tpdelay"))
		{
			player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-teleport-after").replaceAll("%time%", String.valueOf(tp_delay)));
			final int taskId = Bukkit.getScheduler().scheduleSyncDelayedTask(plugin, new FinishTeleport(plugin.getIslandWorld(), player, dest_loc), 20 * tp_delay);

			plugin.addToTeleportList(player, taskId);
		}
		else
		{
			plugin.teleportPlayer(plugin.getSpawnWorld(), player, dest_loc);
			player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-tele-spawn"));
		}

		return true;
	}

	private boolean commandFixHome(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.fixhome")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		SimpleIslandV6 isle = plugin.getPlayerIsland(player.getName());
		if (isle != null)
		{
			if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "fixhome"))
				return plugin.showError(player,
						plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.fixhome", 0))));

			if (plugin.findIslandSpawn(isle))
				player.sendMessage(plugin.getLoc("info-fixhome"));
			else
				player.sendMessage(plugin.getLoc("error-fixhome"));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return true;
	}

	private boolean commandLock(World world, Player player)
	{
		if (!plugin.getConfig().getBoolean("island-lock", false))
			return plugin.showError(player, plugin.getLoc("info-lock-disabled"));
		if (!(player.hasPermission("islandworld.island.lock")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		SimpleIslandV6 isle = plugin.getPlayerIsland(player.getName());
		if (isle != null)
		{
			if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "lock"))
				return plugin.showError(player,
						plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.lock", 0))));

			isle.setProtected(true);
			plugin.showMessage(player, plugin.getLoc("info-lock-ok"));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return true;
	}

	private boolean commandUnLock(World world, Player player)
	{
		if (!plugin.getConfig().getBoolean("island-lock", false))
			return plugin.showError(player, plugin.getLoc("info-lock-disabled"));
		if (!(player.hasPermission("islandworld.island.unlock")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		SimpleIslandV6 isle = plugin.getPlayerIsland(player.getName());
		if (isle != null)
		{
			if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "unlock"))
				return plugin.showError(player,
						plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.unlock", 0))));

			isle.setProtected(false);
			plugin.showMessage(player, plugin.getLoc("info-unlock-ok"));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return true;
	}

	private boolean commandCalc(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.calc")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		if (playerHaveCalcLimit(player))
			return plugin.showError(player, plugin.getLoc("error-calc-delay").replaceAll("%time%", String.valueOf(Config.CALC_LIMIT)));

		SimpleIslandV6 isle = plugin.getPlayerIsland(player.getName());
		if (isle == null)
			isle = plugin.getHelpingIsland(player);

		if (isle != null)
		{
			if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "calc"))
				return plugin.showError(player,
						plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.calc", 0))));

			lastCalc.put(player.getName().toLowerCase(), System.currentTimeMillis());

			int points = plugin.calcIslandPoints(isle);
			isle.setPoints(points);
			plugin.storePoints(isle.getOwner(), points);

			player.sendMessage(ChatColor.AQUA + plugin.getLoc("info-points").replaceAll("%points%", String.valueOf(points)));
		}
		else
			player.sendMessage(ChatColor.RED + plugin.getLoc("error-no-island"));

		return true;
	}

	private boolean commandRank(World world, Player player)
	{
		if (!(player.hasPermission("islandworld.island.rank")))
			return plugin.showError(player, plugin.getLoc("error-no-perms"));

		if (Config.ECONOMY && !Eco.ecoCheck(plugin, player, "rank"))
			return plugin.showError(player,
					plugin.getLoc("eco-failed").replaceAll("%cost%", String.valueOf(plugin.getConfig().getInt("command-cost.rank", 0))));

		final long lastUpdate = plugin.getRankLastUpdate();
		if (System.currentTimeMillis() > (lastUpdate + (60000 * Config.RELOAD_TIME)))
		{
			plugin.debug("Points readed from db");
			plugin.readPoints();
		}
		if (plugin.getRankLastUpdate() <= 0)
			return plugin.showError(player, plugin.getLoc("error-rank-not-updated"));

		final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm");
		final String dateAsString = simpleDateFormat.format(lastUpdate);

		player.sendMessage(ChatColor.GREEN + plugin.getLoc("info-rank"));

		Map<String, Integer> sortedMapAsc = sortByComparator(plugin.getPoints(), false);

		int c = 1;
		for (Entry<String, Integer> en : sortedMapAsc.entrySet())
		{
			player.sendMessage(ChatColor.GRAY + "#" + c + " " + ChatColor.WHITE + en.getKey() + " : " + ChatColor.DARK_GREEN + en.getValue());
			c++;
		}

		player.sendMessage(ChatColor.GREEN + plugin.getLoc("info-rank-last") + " " + dateAsString);
		player.sendMessage(ChatColor.GRAY + plugin.getLoc("info-rank-island"));

		return true;
	}

	private boolean playerHaveCommandLimit(Player player)
	{
		if (Config.TIME_LIMIT > 0 && lastCommand.containsKey(player.getName().toLowerCase()))
		{
			long time = lastCommand.get(player.getName().toLowerCase());
			long diff = (System.currentTimeMillis() - time) / 1000;

			if (diff <= (Config.TIME_LIMIT * 60))
				return true;
		}
		return false;
	}

	private boolean playerHaveCalcLimit(Player player)
	{
		if (Config.CALC_LIMIT > 0 && lastCalc.containsKey(player.getName().toLowerCase()))
		{
			long time = lastCalc.get(player.getName().toLowerCase());
			long diff = (System.currentTimeMillis() - time) / 1000;

			if (diff <= (Config.CALC_LIMIT * 60))
				return true;
		}
		return false;
	}

	private void showUsage(Player player)
	{
		// Player use only /island without params
		player.sendMessage(ChatColor.YELLOW + "Island World v" + plugin.getDescription().getVersion());
		player.sendMessage(ChatColor.AQUA + plugin.getLoc("com-usage"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-pcreate") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-create"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-pdelete") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-delete"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-pinfo") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-info"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-psethome") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-sethome"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-pdelhome") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-delhome"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-phome") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-home"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-pfixhome") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-fixhome"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-pspawn") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-spawn"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-pexpell") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-expell"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-padd") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-add"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-premove") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-remove"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-ptp") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-tp"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-pvisit") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-visit"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-pleave") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-leave"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-plock") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-lock"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-punlock") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-unlock"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-pcalc") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-calc"));
		player.sendMessage(ChatColor.RED + "/" + plugin.getLoc("com-command") + " " + plugin.getLoc("com-prank") + "" + ChatColor.AQUA + " - "
				+ plugin.getLoc("com-rank"));
	}

	private boolean targetIsSafePoint(World world, SimpleIslandV6 isle)
	{
		MyLocation l = isle.getLocation();

		Block above = world.getBlockAt(l.getBlockX(), l.getBlockY() + 1, l.getBlockZ());
		Block targe = world.getBlockAt(l.getBlockX(), l.getBlockY(), l.getBlockZ());
		Block under1 = world.getBlockAt(l.getBlockX(), l.getBlockY() - 1, l.getBlockZ());
		//Block under2 = world.getBlockAt(l.getBlockX(), l.getBlockY() - 2, l.getBlockZ());

		boolean piston = false;
		for (int x = -1; x <= 1; x++)
		{
			for (int y = -1; y <= 1; y++)
			{
				for (int z = -1; z <= 1; z++)
				{
					Block nb = world.getBlockAt(targe.getX() + x, targe.getY() + y, targe.getZ() + z);

					if (nb != null && (nb.getType() == Material.TRAP_DOOR || nb.getType() == Material.PISTON_BASE || nb.getType() == Material.PISTON_EXTENSION || nb.getType() == Material.PISTON_MOVING_PIECE || nb.getType() == Material.PISTON_STICKY_BASE))
					{
						piston = true;
						break;
					}
				}
			}
		}

		if (!piston && above.getType() == Material.AIR && targe.getType() == Material.AIR && under1.getType() != Material.AIR && under1.getType() != Material.LAVA && under1.getType() != Material.WATER && under1.getType() != Material.TRAP_DOOR)
			// && under2.getType() != Material.AIR && under2.getType() != Material.LAVA && under2.getType() != Material.WATER && under2.getType() != Material.TRAP_DOOR)
			return true;
		else
			return false;
	}
	
	private void removeMobs(MyLocation loc)
	{
		// Check Config
		if (plugin.getConfig().getBoolean("remove-mob-on-tp", false))
		{
			int px = loc.getBlockX();
			int py = loc.getBlockY();
			int pz = loc.getBlockZ();

			int count = 0;
			for (int x = -1; x <= 1; x++)
			{
				for (int z = -1; z <= 1; z++)
				{
					Chunk c = plugin.getIslandWorld().getChunkAt(new Location(plugin.getIslandWorld(), px + x * 16, py, pz + z * 16));
					for (Entity e : c.getEntities())
					{
						if (e != null && e instanceof Monster)
						{
							e.remove();
							count++;
						}
					}
				}
			}
			plugin.debug("Removed " + count + " mobs");
		}
	}

	public class FinishTeleport implements Runnable
	{
		private World world;
		private Player player;
		private MyLocation loc;

		public FinishTeleport(World w, Player o, MyLocation l)
		{
			world = w;
			player = o;
			loc = l;
		}

		@Override
		public void run()
		{
			if (player != null && plugin.isOnTeleportList(player) && loc != null)
			{
				player.sendMessage(ChatColor.YELLOW + plugin.getLoc("info-telporting"));
				plugin.teleportPlayer(world, player, loc);

				plugin.removeFromTeleportList(player);
			}
		}
	}

	private static Map<String, Integer> sortByComparator(Map<String, Integer> unsortMap, final boolean order)
	{

		List<Entry<String, Integer>> list = new LinkedList<Entry<String, Integer>>(unsortMap.entrySet());

		// Sorting the list based on values
		Collections.sort(list, new Comparator<Entry<String, Integer>>()
		{
			public int compare(Entry<String, Integer> o1, Entry<String, Integer> o2)
			{
				if (order)
				{
					return o1.getValue().compareTo(o2.getValue());
				}
				else
				{
					return o2.getValue().compareTo(o1.getValue());

				}
			}
		});

		// Maintaining insertion order with the help of LinkedList
		Map<String, Integer> sortedMap = new LinkedHashMap<String, Integer>();
		for (Entry<String, Integer> entry : list)
		{
			sortedMap.put(entry.getKey(), entry.getValue());
		}

		return sortedMap;
	}

}