package atomicstryker.ruins.common;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Random;
import java.util.concurrent.ConcurrentSkipListSet;

import net.minecraft.init.Blocks;
import net.minecraft.world.World;
import net.minecraft.world.biome.BiomeGenBase;

public class RuinGenerator
{

    private final static String fileName = "RuinsPositionsFile.txt";

    private final RuinHandler ruinsHandler;
    private final RuinStats stats;
    private int NumTries = 0, LastNumTries = 0;
    private final int WORLD_MAX_HEIGHT = 256;
    private final ConcurrentSkipListSet<RuinData> registeredRuins;
    private final HashSet<RuinData> sweptNPruned;
    private File ruinsDataFile;

    public RuinGenerator(RuinHandler rh, String worldName)
    {
        ruinsHandler = rh;
        stats = new RuinStats();
        new LinkedList<RuinIBuildable>();
        registeredRuins = new ConcurrentSkipListSet<RuinData>();
        sweptNPruned = new HashSet<RuinData>();
        ruinsDataFile = new File(rh.saveFolder, fileName);

        if (ruinsDataFile.getAbsolutePath().contains(worldName))
        {
            new LoadThread().start();
        }
        else
        {
            System.err.println("Ruins attempted to load invalid worldname " + worldName + " posfile");
        }
    }

    private class LoadThread extends Thread
    {
        @Override
        public void run()
        {
            loadPosFile(ruinsDataFile);
        }
    }

    public void flushPosFile(String worldName)
    {
        if (registeredRuins.isEmpty() || worldName.equals("MpServer"))
        {
            return;
        }

        new FlushThread().start();
    }

    private class FlushThread extends Thread
    {
        @Override
        public void run()
        {
            if (ruinsDataFile.exists())
            {
                ruinsDataFile.delete();
            }

            try
            {
                PrintWriter pw = new PrintWriter(new BufferedWriter(new FileWriter(ruinsDataFile)));
                pw.println("# Ruins data management file. Below, you see all data accumulated by AtomicStrykers Ruins during the last run of this World.");
                pw.println("# Data is noted as follows: Each line stands for one successfull Ruin spawn. Data syntax is:");
                pw.println("# xMin yMin zMin xMax yMax zMax templateName");
                pw.println("# everything but the last value is an integer value. Template name equals the template file name.");
                pw.println("#");
                pw.println("# DO NOT EDIT THIS FILE UNLESS YOU ARE SURE OF WHAT YOU ARE DOING");
                pw.println("#");
                pw.println("# The primary function of this file is to lock areas you do not want Ruins spawning in. Put them here before worldgen.");
                pw.println("# It should also prevent Ruins re-spawning under any circumstances. Areas registered in here block any overlapping new Ruins.");
                pw.println("# Empty lines and those prefixed by '#' are ignored by the parser. Don't save notes in here, file gets wiped upon flushing.");
                pw.println("#");
                for (RuinData r : registeredRuins)
                {
                    pw.println(r.toString());
                    // System.out.println("saved ruin data line ["+r.toString()+"]");
                }

                pw.flush();
                pw.close();
                // System.out.println("Ruins Positions flushed, entries "+registeredRuins.size());
            }
            catch (IOException e)
            {
                e.printStackTrace();
            }
        }
    }

    private void loadPosFile(File file)
    {
        try
        {
            if (!file.exists())
            {
                file.createNewFile();
            }
            int lineNumber = 1;
            BufferedReader br = new BufferedReader(new FileReader(file));
            String line = br.readLine();
            while (line != null)
            {
                line = line.trim();
                if (!line.startsWith("#") && !line.isEmpty())
                {
                    try
                    {
                        registeredRuins.add(new RuinData(line));
                    }
                    catch (Exception e)
                    {
                        System.err.println("Ruins positions file is invalid in line " + lineNumber + ", skipping...");
                    }
                }

                lineNumber++;
                line = br.readLine();
            }
            br.close();
            // System.out.println("Ruins Positions reloaded. Lines "+lineNumber+", entries "+registeredRuins.size());
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    public boolean generateNormal(World world, Random random, int xBase, int j, int zBase)
    {
        // here we're generating 5 chunks behind normal.
        // This should hopefully solve the "wanderer" problem with edge chunks.

        for (int c = 0; c < ruinsHandler.triesPerChunkNormal; c++)
        {
            if (random.nextFloat() * 100 < ruinsHandler.chanceToSpawnNormal)
            {
                /*
                 * ditch the y coord, we'll be coming down from the top,
                 * checking for a suitable square to start.
                 */
                int xMod = random.nextInt(16);
                int zMod = random.nextInt(16);
                int x = xBase + xMod;
                int z = zBase + zMod;
                createBuilding(world, random, x, z, 0, false);
            }
        }
        return true;
    }

    public boolean generateNether(World world, Random random, int xBase, int j, int zBase)
    {
        for (int c = 0; c < ruinsHandler.triesPerChunkNether; c++)
        {
            if (random.nextFloat() * 100 < ruinsHandler.chanceToSpawnNether)
            {
                /*
                 * ditch the y coord, we'll be coming down from the top,
                 * checking for a suitable square to start.
                 */
                int xMod = (random.nextInt(2) == 1 ? random.nextInt(16) : 0 - random.nextInt(16));
                int zMod = (random.nextInt(2) == 1 ? random.nextInt(16) : 0 - random.nextInt(16));
                int x = xBase + xMod;
                int z = zBase + zMod;
                createBuilding(world, random, x, z, 0, true);
            }
        }
        return true;
    }

    private void createBuilding(World world, Random random, int x, int z, int minDistance, boolean Nether)
    {
        int rotate = random.nextInt(4);
        BiomeGenBase biome = world.getBiomeGenForCoordsBody(x, z);
        int biomeID = biome.biomeID;
        int nextMinDistance = 0;

        if (ruinsHandler.useGeneric(random, biomeID))
        {
            biomeID = RuinsMod.BIOME_NONE;
        }
        stats.biomes[biomeID]++;

        RuinIBuildable ruinTemplate = ruinsHandler.getTemplate(random, biomeID);
        if (ruinTemplate == null)
        {
            biomeID = RuinsMod.BIOME_NONE;
            ruinTemplate = ruinsHandler.getTemplate(random, biomeID);

            if (ruinTemplate == null)
            {
                return;
            }
        }

        NumTries++;

        if (minDistance != 0)
        {
            stats.SiteTries++;
            // tweak the x and z from the Min Distance, minding the bounding box
            minDistance += random.nextInt(3) + ruinTemplate.getMinDistance();
            x += (random.nextInt(2) == 1 ? 0 - minDistance : minDistance);
            z += (random.nextInt(2) == 1 ? 0 - minDistance : minDistance);
        }

        int y = findSuitableY(world, ruinTemplate, x, z, Nether);
        if (y > 0)
        {
            if (willOverlap(ruinTemplate, x, y, z, rotate))
            {
                // try again.
                int xTemp = getRandomAdjustment(random, x, minDistance);
                int zTemp = getRandomAdjustment(random, z, minDistance);
                if (willOverlap(ruinTemplate, xTemp, y, zTemp, rotate))
                {
                    // last chance
                    xTemp = getRandomAdjustment(random, x, minDistance);
                    zTemp = getRandomAdjustment(random, z, minDistance);
                    if (willOverlap(ruinTemplate, xTemp, y, zTemp, rotate))
                    {
                        stats.BoundingBoxFails++;
                        // System.out.println("Bounding Box fail "+stats.BoundingBoxFails);
                        return;
                    }
                    x = xTemp;
                    z = zTemp;
                }
                else
                {
                    x = xTemp;
                    z = zTemp;
                }
            }

            if (ruinTemplate.checkArea(world, x, y, z, rotate) && checkMinDistance(ruinTemplate.getRuinData(x, y, z, rotate)))
            {
                if (!ruinsHandler.disableLogging)
                {
                    if (minDistance != 0)
                    {
                        System.out.printf("Creating ruin %s of Biome %s as part of a site at [%d|%d|%d]\n", ruinTemplate.getName(), biome.biomeName,
                                x, y, z);
                    }
                    else
                    {
                        System.out.printf("Creating ruin %s of Biome %s at [%d|%d|%d]\n", ruinTemplate.getName(), biome.biomeName, x, y, z);
                    }
                }
                stats.NumCreated++;

                ruinTemplate.doBuild(world, random, x, y, z, rotate);
                registeredRuins.add(ruinTemplate.getRuinData(x, y, z, rotate));
                nextMinDistance = ruinTemplate.getMinDistance();
                if (ruinTemplate.isUnique())
                {
                    ruinsHandler.removeTemplate(ruinTemplate, biomeID);
                    try
                    {
                        ruinsHandler.writeExclusions(ruinsHandler.saveFolder);
                    }
                    catch (Exception e)
                    {
                        System.err.println("Could not write exclusions for world: " + ruinsHandler.saveFolder);
                        e.printStackTrace();
                    }
                }
            }
            else
            {
                // System.out.println("Area check fail");
                return;
            }
            if (Nether)
            {
                if (random.nextFloat() * 100 < ruinsHandler.chanceForSiteNether)
                {
                    createBuilding(world, random, x, z, nextMinDistance, true);
                }
            }
            else
            {
                if (random.nextFloat() * 100 < ruinsHandler.chanceForSiteNormal)
                {
                    createBuilding(world, random, x, z, nextMinDistance, false);
                }
            }
        }
        else
        {
            // System.out.println("y fail");
        }

        if (NumTries > (LastNumTries + 1000))
        {
            LastNumTries = NumTries;
            printStats();
        }
    }

    private void printStats()
    {
        if (!ruinsHandler.disableLogging)
        {
            int total =
                    stats.NumCreated + stats.BadBlockFails + stats.LevelingFails + stats.CutInFails + stats.OverhangFails + stats.NoAirAboveFails
                            + stats.BoundingBoxFails;
            System.out.println("Current Stats:");
            System.out.println("    Total Tries:                 " + total);
            System.out.println("    Number Created:              " + stats.NumCreated);
            System.out.println("    Site Tries:                  " + stats.SiteTries);
            System.out.println("    Within Another Bounding Box: " + stats.BoundingBoxFails);
            System.out.println("    Bad Blocks:                  " + stats.BadBlockFails);
            System.out.println("    No Leveling:                 " + stats.LevelingFails);
            System.out.println("    No Cut-In:                   " + stats.CutInFails);

            for (int i = 0; i < stats.biomes.length; i++)
            {
                if (stats.biomes[i] != 0)
                {
                    if (i != RuinsMod.BIOME_NONE)
                        System.out.println(BiomeGenBase.func_150565_n()[i].biomeName + ": " + stats.biomes[i] + " Biome building attempts");
                    else
                        System.out.println("Any-Biome: " + stats.biomes[i] + " building attempts");
                }
            }

            System.out.println();
        }
    }

    private int getRandomAdjustment(Random random, int base, int minDistance)
    {
        return random.nextInt(8) - random.nextInt(8) + (random.nextInt(2) == 1 ? 0 - minDistance : minDistance);
    }
    
    /**
     * Executes a Sweep n Prune algorithm by only putting RuinData sets with at all possible collisions
     * into a subset, which then is to be used instead of the full RuinData set for collision detection
     * 
     * @param collider RuinData instance to find possible collisions for
     * @return reference to reused sweptNPruned HashSet
     */
    private HashSet<RuinData> sweptAndPrunedSetColliding(RuinData collider)
    {
        sweptNPruned.clear();
        RuinData other = registeredRuins.floor(collider);
        while (other != null && collider.collisionLowerBoundsPossible(other))
        {
            if (sweptNPruned.add(other))
            {
                other = registeredRuins.lower(other);
            }
            else
            {
                break;
            }
        }
        other = registeredRuins.ceiling(collider);
        while (other != null && collider.collisionHigherBoundsPossible(other))
        {
            if (sweptNPruned.add(other))
            {
                other = registeredRuins.higher(other);
            }
            else
            {
                break;
            }
        }
        return sweptNPruned;
    }

    private boolean willOverlap(RuinIBuildable r, int x, int y, int z, int rotate)
    {
        RuinData current = r.getRuinData(x, y, z, rotate);
        for (RuinData rd : sweptAndPrunedSetColliding(current))
        {
            if (rd.collides(current))
            {
                return true;
            }
        }
        return false;
    }
    
    private boolean checkMinDistance(RuinData ruinData)
    {
        // refuse Ruins spawning too close to each other
        float minDistTemplate = ruinsHandler.templateInstancesMinDistance;
        minDistTemplate = minDistTemplate*minDistTemplate; // square it
        float minDistRuins = ruinsHandler.anyRuinsMinDistance;
        minDistRuins = minDistRuins*minDistRuins; //squared
        
        for (RuinData r : registeredRuins)
        {
            if (r.name.equals(ruinData.name))
            {
                if (r.getDistanceSqTo(ruinData) < minDistTemplate)
                {
                    return false;
                }
            }
            else
            {
                if (r.getDistanceSqTo(ruinData) < minDistRuins)
                {
                    return false;
                }
            }
        }
        
        return true;
    }

    private int findSuitableY(World world, RuinIBuildable r, int x, int z, boolean Nether)
    {
        if (Nether)
        {
            /*
             * The Nether has an entirely different topography so we'll use two
             * methods in a semi-random fashion (since we're not getting the
             * random here)
             */
            if ((x % 2 == 1) ^ (z % 2 == 1))
            {
                // from the top. Find the first air block from the ceiling
                for (int y = WORLD_MAX_HEIGHT - 1; y > -1; y--)
                {
                    if (world.func_147439_a(x, y, z) == Blocks.air)
                    {
                        // now find the first non-air block from here
                        for (; y > -1; y--)
                        {
                            if (!r.isAir(world.func_147439_a(x, y, z)))
                            {
                                if (r.isAcceptable(world, x, y, z))
                                {
                                    return y;
                                }
                            }
                        }
                    }
                }
            }
            else
            {
                // from the bottom. find the first air block from the floor
                for (int y = 0; y < WORLD_MAX_HEIGHT; y++)
                {
                    if (!r.isAir(world.func_147439_a(x, y, z)))
                    {
                        if (r.isAcceptable(world, x, y - 1, z))
                        {
                            return y - 1;
                        }
                    }
                }
            }
            return -1;
        }
        else
        {
            for (int y = WORLD_MAX_HEIGHT - 1; y > -1; y--)
            {
                if (r.isAcceptable(world, x, y, z))
                {
                    return y;
                }
                if (!r.isAir(world.func_147439_a(x, y, z)))
                {
                    return -1;
                }
            }
        }
        return -1;
    }
}